const nodemailer = require("nodemailer");
const getEnvConfig = require("./Config");

// Create reusable transporter object using SMTP transport
const createTransporter = () => {
  const smtpHost = getEnvConfig("SMTP_HOST") || "smtp.gmail.com";
  const smtpPort = parseInt(getEnvConfig("SMTP_PORT") || "587");
  const smtpUser = getEnvConfig("SMTP_USER");
  const smtpPassword = getEnvConfig("SMTP_PASSWORD");
  const smtpFrom = getEnvConfig("SMTP_FROM") || smtpUser;

  if (!smtpUser || !smtpPassword) {
    console.warn("SMTP credentials not configured. Email sending will be disabled.");
    return null;
  }

  return nodemailer.createTransport({
    host: smtpHost,
    port: smtpPort,
    secure: smtpPort === 465, // true for 465, false for other ports
    auth: {
      user: smtpUser,
      pass: smtpPassword,
    },
  });
};

/**
 * Send email verification email
 * @param {string} to - Recipient email address
 * @param {string} name - Recipient name
 * @param {string} verificationToken - Verification token
 * @param {string} baseUrl - Base URL of the application
 */
const sendVerificationEmail = async (to, name, verificationToken, baseUrl) => {
  try {
    const transporter = createTransporter();
    if (!transporter) {
      console.error("Email transporter not configured. Cannot send verification email.");
      return { success: false, error: "Email service not configured" };
    }

    const verificationUrl = `${baseUrl}/verify-email?token=${verificationToken}`;
    const smtpFrom = getEnvConfig("SMTP_FROM") || getEnvConfig("SMTP_USER");

    const mailOptions = {
      from: `"${getEnvConfig("APP_NAME") || "Large App"}" <${smtpFrom}>`,
      to: to,
      subject: "Verify Your Email Address",
      html: `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Verify Your Email</title>
        </head>
        <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background-color: #f4f4f4; padding: 20px; border-radius: 5px;">
            <h1 style="color: #333; text-align: center;">Email Verification</h1>
            <p>Hello ${name},</p>
            <p>Thank you for registering with us! Please verify your email address by clicking the button below:</p>
            <div style="text-align: center; margin: 30px 0;">
              <a href="${verificationUrl}" 
                 style="background-color: #007bff; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; display: inline-block;">
                Verify Email Address
              </a>
            </div>
            <p>Or copy and paste this link into your browser:</p>
            <p style="word-break: break-all; color: #007bff;">${verificationUrl}</p>
            <p>This verification link will expire in 24 hours.</p>
            <p>If you did not create an account, please ignore this email.</p>
            <hr style="border: none; border-top: 1px solid #eee; margin: 20px 0;">
            <p style="font-size: 12px; color: #666;">This is an automated email, please do not reply.</p>
          </div>
        </body>
        </html>
      `,
      text: `
        Hello ${name},
        
        Thank you for registering with us! Please verify your email address by clicking the link below:
        
        ${verificationUrl}
        
        This verification link will expire in 24 hours.
        
        If you did not create an account, please ignore this email.
      `,
    };

    const info = await transporter.sendMail(mailOptions);
    console.log("Verification email sent:", info.messageId);
    return { success: true, messageId: info.messageId };
  } catch (error) {
    console.error("Error sending verification email:", error);
    return { success: false, error: error.message };
  }
};

/**
 * Send password reset email
 * @param {string} to - Recipient email address
 * @param {string} name - Recipient name
 * @param {string} resetToken - Password reset token
 * @param {string} baseUrl - Base URL of the application
 */
const sendPasswordResetEmail = async (to, name, resetToken, baseUrl) => {
  try {
    const transporter = createTransporter();
    if (!transporter) {
      console.error("Email transporter not configured. Cannot send password reset email.");
      return { success: false, error: "Email service not configured" };
    }

    const resetUrl = `${baseUrl}/reset-password?token=${resetToken}`;
    const smtpFrom = getEnvConfig("SMTP_FROM") || getEnvConfig("SMTP_USER");

    const mailOptions = {
      from: `"${getEnvConfig("APP_NAME") || "Large App"}" <${smtpFrom}>`,
      to: to,
      subject: "Reset Your Password",
      html: `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Reset Your Password</title>
        </head>
        <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background-color: #f4f4f4; padding: 20px; border-radius: 5px;">
            <h1 style="color: #333; text-align: center;">Password Reset Request</h1>
            <p>Hello ${name},</p>
            <p>We received a request to reset your password. Click the button below to reset it:</p>
            <div style="text-align: center; margin: 30px 0;">
              <a href="${resetUrl}" 
                 style="background-color: #dc3545; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; display: inline-block;">
                Reset Password
              </a>
            </div>
            <p>Or copy and paste this link into your browser:</p>
            <p style="word-break: break-all; color: #007bff;">${resetUrl}</p>
            <p>This reset link will expire in 1 hour.</p>
            <p>If you did not request a password reset, please ignore this email. Your password will remain unchanged.</p>
            <hr style="border: none; border-top: 1px solid #eee; margin: 20px 0;">
            <p style="font-size: 12px; color: #666;">This is an automated email, please do not reply.</p>
          </div>
        </body>
        </html>
      `,
      text: `
        Hello ${name},
        
        We received a request to reset your password. Click the link below to reset it:
        
        ${resetUrl}
        
        This reset link will expire in 1 hour.
        
        If you did not request a password reset, please ignore this email. Your password will remain unchanged.
      `,
    };

    const info = await transporter.sendMail(mailOptions);
    console.log("Password reset email sent:", info.messageId);
    return { success: true, messageId: info.messageId };
  } catch (error) {
    console.error("Error sending password reset email:", error);
    return { success: false, error: error.message };
  }
};

module.exports = {
  sendVerificationEmail,
  sendPasswordResetEmail,
};

